<?php

namespace App\Http\Controllers\Karyawan;

use Carbon\Carbon;
use App\Models\Cuti;
use App\Models\Absensi;
use App\Models\Karyawan;
use App\Models\AjukanShift;
use App\Models\JadwalShift;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    /**
     * Get shift aktif untuk karyawan berdasarkan tanggal
     */
    private function getActiveShift($karyawan, $tanggal)
    {
        $departemen_id = $karyawan->departemen_id;

        // 1. Cek apakah ada pengajuan shift SEMENTARA yang disetujui dan masih aktif
        $shiftSementara = AjukanShift::where('departemen_id', $departemen_id)
            ->where('jenis', 'sementara')
            ->where('status', 'disetujui')
            ->where('tanggal_mulai', '<=', $tanggal)
            ->where('tanggal_selesai', '>=', $tanggal)
            ->with('shiftBaru')
            ->orderBy('created_at', 'desc')
            ->first();

        if ($shiftSementara) {
            return $shiftSementara->shiftBaru;
        }

        // 2. Jika tidak ada shift sementara, ambil shift default dari jadwal_shift yang aktif
        $jadwalShift = JadwalShift::where('departemen_id', $departemen_id)
            ->where('is_active', true)
            ->where('tanggal_mulai', '<=', $tanggal)
            ->where(function ($q) use ($tanggal) {
                $q->whereNull('tanggal_selesai')
                  ->orWhere('tanggal_selesai', '>=', $tanggal);
            })
            ->with('shift')
            ->orderBy('tanggal_mulai', 'desc')
            ->first();

        if ($jadwalShift) {
            return $jadwalShift->shift;
        }

        return null;
    }

    /**
     * Display dashboard
     */
    public function index()
    {
        $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();
        $today = Carbon::now()->toDateString();

        // Get shift aktif hari ini (bisa default atau sementara)
        $shiftData = $this->getActiveShift($karyawan, $today);

        // Absensi hari ini
        $absensiToday = Absensi::where('karyawan_id', $karyawan->id)
            ->whereDate('tanggal', $today)
            ->first();

        // Jika tidak ada absensi hari ini, cek kemarin untuk shift lintas hari
        if (!$absensiToday && $shiftData && $shiftData->lintas_hari) {
            $yesterday = Carbon::yesterday()->toDateString();
            $yesterdayShift = $this->getActiveShift($karyawan, $yesterday);

            if ($yesterdayShift && $yesterdayShift->lintas_hari) {
                $absensiToday = Absensi::where('karyawan_id', $karyawan->id)
                    ->whereDate('tanggal', $yesterday)
                    ->first();
            }
        }

        // Statistik bulan ini
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $stats = [
            'hadir' => Absensi::where('karyawan_id', $karyawan->id)
                ->whereMonth('tanggal', $currentMonth)
                ->whereYear('tanggal', $currentYear)
                ->where('status', 'hadir')
                ->count(),

            'terlambat' => Absensi::where('karyawan_id', $karyawan->id)
                ->whereMonth('tanggal', $currentMonth)
                ->whereYear('tanggal', $currentYear)
                ->where('status', 'terlambat')
                ->count(),

            'alpa' => Absensi::where('karyawan_id', $karyawan->id)
                ->whereMonth('tanggal', $currentMonth)
                ->whereYear('tanggal', $currentYear)
                ->where('status', 'alpha')
                ->count(),
        ];

        // ========== TAMBAHAN: CALENDAR DATA ==========
        $calendarData = $this->generateCalendarData($karyawan->id);

        // Jika shift tidak ada, set default values
        if (!$shiftData) {
            return view('karyawan.dashboard', [
                'shift' => 'Belum Ditentukan',
                'jamMasuk' => '--:--',
                'jamPulang' => '--:--',
                'toleransi' => 0,
                'lintasHari' => false,
                'absensiToday' => $absensiToday,
                'stats' => $stats,
                'calendarData' => $calendarData // ← BARU
            ])->with('error', 'Shift departemen Anda belum ditentukan. Hubungi admin!');
        }

        // Return view dengan data lengkap
        return view('karyawan.dashboard', [
            'shift' => $shiftData->kode,
            'jamMasuk' => substr($shiftData->jam_masuk, 0, 5),
            'jamPulang' => substr($shiftData->jam_pulang, 0, 5),
            'toleransi' => $shiftData->toleransi_menit,
            'lintasHari' => $shiftData->lintas_hari,
            'absensiToday' => $absensiToday,
            'stats' => $stats,
            'calendarData' => $calendarData // ← BARU
        ]);
    }

    /**
     * ========== BARU: Generate calendar data untuk mini calendar ==========
     * Return: Array dengan key tanggal (Y-m-d) dan value status
     */
    private function generateCalendarData($karyawanId)
    {
        $calendarData = [];

        // Ambil 3 bulan: bulan lalu, bulan ini, bulan depan
        $startDate = now()->startOfMonth()->subMonth();
        $endDate = now()->endOfMonth()->addMonth();

        // 1. Ambil data absensi
        $absensi = Absensi::where('karyawan_id', $karyawanId)
            ->whereBetween('tanggal', [$startDate, $endDate])
            ->get()
            ->keyBy(function($item) {
                return Carbon::parse($item->tanggal)->format('Y-m-d');
            });

        // 2. Ambil data cuti yang disetujui
        $cuti = Cuti::where('karyawan_id', $karyawanId)
            ->where('status', 'disetujui')
            ->where(function($query) use ($startDate, $endDate) {
                $query->whereBetween('tanggal_mulai', [$startDate, $endDate])
                      ->orWhereBetween('tanggal_selesai', [$startDate, $endDate])
                      ->orWhere(function($q) use ($startDate, $endDate) {
                          $q->where('tanggal_mulai', '<=', $startDate)
                            ->where('tanggal_selesai', '>=', $endDate);
                      });
            })
            ->get();

        // Loop setiap hari dalam range
        $current = $startDate->copy();
        while ($current <= $endDate) {
            $dateStr = $current->format('Y-m-d');

            // Cek apakah tanggal ini ada di data absensi
            if (isset($absensi[$dateStr])) {
                $status = $absensi[$dateStr]->status;

                // Map status ke calendar class
                switch($status) {
                    case 'hadir':
                        $calendarData[$dateStr] = 'hadir';
                        break;
                    case 'terlambat':
                        $calendarData[$dateStr] = 'terlambat';
                        break;
                    case 'alpha':
                    case 'alpa':
                        $calendarData[$dateStr] = 'alpa';
                        break;
                    case 'sakit':
                        $calendarData[$dateStr] = 'sakit';
                        break;
                    case 'izin':
                        $calendarData[$dateStr] = 'izin';
                        break;
                    case 'cuti':
                        $calendarData[$dateStr] = 'cuti';
                        break;
                    default:
                        // Jika ada status lain, tampilkan sebagai libur
                        $calendarData[$dateStr] = 'libur';
                }
            }
            // Cek apakah tanggal ini dalam periode cuti
            else {
                foreach ($cuti as $c) {
                    $cutiStart = Carbon::parse($c->tanggal_mulai);
                    $cutiEnd = Carbon::parse($c->tanggal_selesai);

                    if ($current->between($cutiStart, $cutiEnd)) {
                        $calendarData[$dateStr] = 'cuti';
                        break;
                    }
                }
            }

            // Tandai hari Minggu sebagai libur (jika belum ada status)
            if (!isset($calendarData[$dateStr]) && $current->dayOfWeek === 0) {
                $calendarData[$dateStr] = 'libur';
            }

            $current->addDay();
        }

        return $calendarData;
    }

    /**
     * ========== BARU (OPTIONAL): Get calendar data via AJAX ==========
     * Untuk load calendar dinamis saat user ganti bulan
     */
    public function getCalendarData(Request $request)
    {
        $month = $request->input('month', now()->month);
        $year = $request->input('year', now()->year);
        $karyawan = Karyawan::where('user_id', Auth::id())->firstOrFail();

        $startDate = Carbon::create($year, $month, 1)->startOfMonth();
        $endDate = Carbon::create($year, $month, 1)->endOfMonth();

        $calendarData = [];

        // Ambil data absensi bulan ini
        $absensi = Absensi::where('karyawan_id', $karyawan->id)
            ->whereBetween('tanggal', [$startDate, $endDate])
            ->get()
            ->keyBy(function($item) {
                return Carbon::parse($item->tanggal)->format('Y-m-d');
            });

        // Ambil data cuti yang overlap dengan bulan ini
        $cuti = Cuti::where('karyawan_id', $karyawan->id)
            ->where('status', 'disetujui')
            ->where(function($query) use ($startDate, $endDate) {
                $query->whereBetween('tanggal_mulai', [$startDate, $endDate])
                      ->orWhereBetween('tanggal_selesai', [$startDate, $endDate])
                      ->orWhere(function($q) use ($startDate, $endDate) {
                          $q->where('tanggal_mulai', '<=', $startDate)
                            ->where('tanggal_selesai', '>=', $endDate);
                      });
            })
            ->get();

        $current = $startDate->copy();
        while ($current <= $endDate) {
            $dateStr = $current->format('Y-m-d');

            if (isset($absensi[$dateStr])) {
                $status = $absensi[$dateStr]->status;

                // Handle berbagai kemungkinan nama status
                switch($status) {
                    case 'alpha':
                    case 'alpa':
                        $calendarData[$dateStr] = 'alpa';
                        break;
                    default:
                        $calendarData[$dateStr] = $status;
                }
            } else {
                // Cek cuti
                foreach ($cuti as $c) {
                    if ($current->between(Carbon::parse($c->tanggal_mulai), Carbon::parse($c->tanggal_selesai))) {
                        $calendarData[$dateStr] = 'cuti';
                        break;
                    }
                }

                // Minggu = libur
                if (!isset($calendarData[$dateStr]) && $current->dayOfWeek === 0) {
                    $calendarData[$dateStr] = 'libur';
                }
            }

            $current->addDay();
        }

        return response()->json($calendarData);
    }
}
